/*
  A scanner for a simple language allowing the following two types of
  variable assignment statements.

  String var1 = "hello world 1 + 2";
  int var2 = 1 - (2 + 3);
  int var3 = 4 + var2;

  Exercises are marked by Attention signs: /!\.
*/

%%

//------------------------------------------------------------------------------
// This code will be included verbatim into the scanner class.

%{
    // We will use this to put together the characters of string
    // literals.
    private StringBuilder sbuilder = new StringBuilder();

    // The line and the character at which the string literal starts.
    private int sline = 0;
    private int sstart = 0;
%}
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Directives that control the behaviour of JFlex.

%line              // Turn on line count.
%char              // Turn on character count.
%unicode           // Use Unicode.
%debug             // Generate a debug main class that can run our scanner.
%type ArithToken   // The scanning function will return instances of ArithToken.
//------------------------------------------------------------------------------

    
//------------------------------------------------------------------------------
// Aliases for regular expressions we will use in scanning tokens.

newline=\r|\n|\r\n
whitespace=[ \t]

digit=[0-9]
letter=[A-Za-z]

identifier={letter}({digit}|{letter})*
//------------------------------------------------------------------------------
    

//------------------------------------------------------------------------------
// Additional states used to scan complex expressions.
     
// We will need this extra state to scan string literals.
%state STRING

//   /!\ EXERCISE: Define strings as a regular expressions.

//------------------------------------------------------------------------------

%%


// Here we will define the rules for splitting the supplied string
// into tokens.

// When the scanner is state <YYINITIAL> and finds the keyword
// "string", do the following:
<YYINITIAL> "String" {
    // create a new token instance and fill it with information.
    return new ArithToken(
			  ArithToken.Type.STRING_TYPE,
			  yytext(), // returns the contents of the token, "string" in our case.
			  yyline,   // the current line number.
			  yychar,   // the position in the string where the token starts.
			  yychar + "String".length() // the position where the token ends.
			  );
}

// Same thing for the keyword "int".
<YYINITIAL> "int" {
    return new ArithToken(ArithToken.Type.INT_TYPE, yytext(), yyline, yychar, yychar + "int".length());
}

// We can group several actions for one and the same state of the
// scanner.
<YYINITIAL> {
    "=" { return new ArithToken(ArithToken.Type.EQUALS, yytext(), yyline, yychar, yychar + 1); }
    "+" { return new ArithToken(ArithToken.Type.PLUS,   yytext(), yyline, yychar, yychar + 1); }
    "-" { return new ArithToken(ArithToken.Type.MINUS,  yytext(), yyline, yychar, yychar + 1); }
    "(" { return new ArithToken(ArithToken.Type.LPAREN, yytext(), yyline, yychar, yychar + 1); }
    ")" { return new ArithToken(ArithToken.Type.RPAREN, yytext(), yyline, yychar, yychar + 1); }
    ";" { return new ArithToken(ArithToken.Type.SCOLON, yytext(), yyline, yychar, yychar + 1); }

    // We can use regular expressions or pseudonyms of regular
    // expressions.
    {identifier} {
	return new ArithToken(ArithToken.Type.IDENTIFIER, yytext(), yyline, yychar, yychar + yylength());
    }

    {digit}+ {
	return new ArithToken(ArithToken.Type.NUMBER, yytext(), yyline, yychar, yychar + yylength());
    }

    
    // When we run into a quote while in state <YYINITIAL>, we know it
    // is an opening quote and we should start processing a string
    // literal.
    \" {
	// Reset our StringBuilder.
	sbuilder.setLength(0);

	// Remember where the string literal starts.
	sline = yyline;
	sstart = yychar;

	// Move the scanner into state <STRING>.
	yybegin(STRING);
    }

    ({whitespace}|{newline})+ {
	// Nobody cares about whitespace.
    }
}


// This is what happens when the scanner is in state <STRING> (i.e. it
// is looking over a string literal).
<STRING> {
    // When we find another quote, we know it is the closing one, so
    // we can finalise the construction of the string and of the
    // token, and switch back to normal scanning mode (state
    // <YYINITIAL>).
    \" {
	// Switch back to normal state.
	yybegin(YYINITIAL);

	// Return the token containing the string literal.
	return new ArithToken(ArithToken.Type.STRING, sbuilder.toString(), sline, sstart, sstart + sbuilder.length());
    }

    // When we find anything but a closing quote, just append it to
    // the string.
    [^\"]+ {
	sbuilder.append(yytext());
    }

    //  /!\  Handle the escaped character sequences.
}
